# 機能設計書 4-ビット演算（Bitwise Operations）

## 概要

本ドキュメントは、TensorFlowのコアフレームワークにおけるビット演算（Bitwise Operations）機能の設計を記述する。ビット単位のAND・OR・XOR・シフトなどの演算を提供する。

### 本機能の処理概要

ビット演算機能は、整数型テンソルに対するビット単位の論理演算およびシフト演算を提供する。ハッシュ計算、ビットマスク操作、低レベルデータ操作などに使用される基本的なビット操作群である。

**業務上の目的・背景**：ハッシュ関数の実装、フラグ管理、ビットパッキングによるメモリ効率化、暗号学的操作の基盤など、低レベルのデータ操作で必要となるビット演算を提供する。機械学習では特徴量のハッシュ化やビット量子化で使用される。

**機能の利用シーン**：(1) 特徴量のビットハッシュ化、(2) ビットマスクによるフラグ管理、(3) ビットシフトによる効率的な数値演算、(4) PopulationCountによるビットカウント、(5) ビット反転操作。

**主要な処理内容**：
1. Invert: ビット単位の反転（NOT）
2. BitwiseAnd: ビット単位のAND（可換）
3. BitwiseOr: ビット単位のOR（可換）
4. BitwiseXor: ビット単位のXOR（可換）
5. LeftShift: 左ビットシフト
6. RightShift: 右ビットシフト
7. PopulationCount: セットビット数のカウント

**関連システム・外部連携**：C++のgen_bitwise_opsカーネルを通じて実行される。すべての操作は非微分可能（NotDifferentiable）として登録されている。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はコアAPIのため直接的な画面関連はなし |

## 機能種別

計算処理（ビット演算処理）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| x | tf.Tensor | Yes | 第1オペランド | int8/int16/int32/int64/uint8/uint16/uint32/uint64型 |
| y | tf.Tensor | 二項演算時 | 第2オペランド | xとブロードキャスト互換の整数型 |

### 入力データソース

Python API呼び出し時の引数として入力される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | tf.Tensor | 演算結果テンソル（入力と同じ型、PopulationCountはuint8） |

### 出力先

呼び出し元のPythonコードに返却される。

## 処理フロー

### 処理シーケンス

```
1. Python API呼び出し（tf.bitwise名前空間）
   └─ 入力テンソルの整数型検証
2. ブロードキャスト形状計算（二項演算時）
   └─ BroadcastBinaryOpShapeFn
3. gen_bitwise_ops経由でC++カーネル呼び出し
4. 結果テンソル返却
```

### フローチャート

```mermaid
flowchart TD
    A[tf.bitwise API呼び出し] --> B[入力型チェック:整数型]
    B --> C{単項演算?}
    C -->|Yes: Invert/PopCount| D[UnchangedShape推論]
    C -->|No: AND/OR/XOR/Shift| E[BroadcastBinaryOpShapeFn]
    D --> F[C++カーネル実行]
    E --> F
    F --> G[結果テンソル返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 整数型制約 | すべてのビット演算は整数型のみサポート | 常時 |
| BR-02 | 非微分可能 | ビット演算は微分不可能として登録 | 常時 |
| BR-03 | 可換性 | AND, OR, XORは可換演算（SetIsCommutative） | 対象操作時 |
| BR-04 | ブロードキャスト | 二項演算はブロードキャスト規則に従う | 二項演算時 |

### 計算ロジック

- Invert: y = ~x（ビット単位の補数）
- BitwiseAnd: z = x & y
- BitwiseOr: z = x | y
- BitwiseXor: z = x ^ y
- LeftShift: z = x << y
- RightShift: z = x >> y（算術シフト）
- PopulationCount: y = popcount(x)（1ビットの数をカウント）

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作は発生しない |

### テーブル別操作詳細

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| TypeError | 型エラー | 非整数型テンソルを入力 | int型テンソルを使用する |
| InvalidArgumentError | ブロードキャストエラー | 形状が非互換 | 入力形状を揃える |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- 要素ごとの演算であり、GPU上でデータ並列に実行されること
- メモリオーバーヘッドなく、入力テンソルと同じサイズの出力を生成

## セキュリティ考慮事項

- 特段のセキュリティリスクはないが、暗号学的用途では適切な乱数生成と組み合わせること

## 備考

- bitwise_ops.pyは非常にシンプルで、gen_bitwise_opsのwildcard importとNotDifferentiable登録のみを行う
- サポートされるデータ型: int8, int16, int32, int64, uint8, uint16, uint32, uint64

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

ビット演算がサポートする整数型を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | bitwise_ops.cc | `tensorflow/core/ops/bitwise_ops.cc` | サポートされる型（Attr "T"） |

**読解のコツ**: ビット演算のC++定義はマクロ（BINARY_BITWISE, BINARY_BITWISE_COMMUTATIVE）を使用して二項演算を簡潔に定義している。

#### Step 2: エントリーポイントを理解する

Python側のバインディングとNotDifferentiable登録を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | bitwise_ops.py | `tensorflow/python/ops/bitwise_ops.py` | gen_bitwise_opsのimportとNotDifferentiable登録 |

**主要処理フロー**:
1. **24行目**: `from tensorflow.python.ops.gen_bitwise_ops import *` - 全opのインポート
2. **27行目**: `ops.NotDifferentiable("BitwiseAnd")` - AND演算の非微分登録
3. **28行目**: `ops.NotDifferentiable("BitwiseOr")` - OR演算の非微分登録
4. **29行目**: `ops.NotDifferentiable("BitwiseXor")` - XOR演算の非微分登録
5. **30行目**: `ops.NotDifferentiable("Invert")` - 反転演算の非微分登録
6. **31行目**: `ops.NotDifferentiable("PopulationCount")` - ポップカウントの非微分登録
7. **32行目**: `ops.NotDifferentiable("LeftShift")` - 左シフトの非微分登録
8. **33行目**: `ops.NotDifferentiable("RightShift")` - 右シフトの非微分登録

#### Step 3: C++オペレーション定義を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | bitwise_ops.cc | `tensorflow/core/ops/bitwise_ops.cc` | 全7 opの定義 |

**主要処理フロー**:
- **23-27行目**: `Invert` op - 単項演算、UnchangedShape
- **29-34行目**: `BINARY_BITWISE` マクロ - 非可換二項演算テンプレート
- **36-42行目**: `BINARY_BITWISE_COMMUTATIVE` マクロ - 可換二項演算テンプレート
- **44-48行目**: `PopulationCount` op - uint8出力
- **50行目**: `BitwiseAnd` op
- **52行目**: `BitwiseOr` op
- **54行目**: `BitwiseXor` op
- **56行目**: `LeftShift` op
- **58行目**: `RightShift` op

### プログラム呼び出し階層図

```
tf.bitwise.bitwise_and(x, y)
    |
    +-- gen_bitwise_ops.bitwise_and(x, y)
            |
            +-- C++ BitwiseAnd kernel
                    |
                    +-- BroadcastBinaryOpShapeFn (形状推論)
                    +-- 要素ごとビットAND演算
```

### データフロー図

```
[入力]                    [処理]                        [出力]

整数テンソル x, y  -->  bitwise_ops.py             -->  結果テンソル z
(int8-int64,           (gen_bitwise_ops呼び出し)       (同型)
 uint8-uint64)              |
                    bitwise_ops.cc
                    (BroadcastBinaryOpShapeFn)
                            |
                    C++ Kernel
                    (ビット演算実行)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| bitwise_ops.py | `tensorflow/python/ops/bitwise_ops.py` | ソース | Python API・NotDifferentiable登録 |
| bitwise_ops.cc | `tensorflow/core/ops/bitwise_ops.cc` | ソース | C++ op定義 |
| gen_bitwise_ops.py | `tensorflow/python/ops/gen_bitwise_ops.py` | 自動生成 | Python バインディング |
| common_shape_fns.h | `tensorflow/core/framework/common_shape_fns.h` | ヘッダ | 形状推論関数 |
| numeric_op.h | `tensorflow/core/framework/numeric_op.h` | ヘッダ | 数値op基盤 |
